<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require "config.php";


/* ----------------------------------------------------
    READ TELEGRAM UPDATE
---------------------------------------------------- */
$raw = file_get_contents("php://input");
file_put_contents(__DIR__ . "/bot_input_log.json", date('c') . " - " . $raw . PHP_EOL, FILE_APPEND);

$update = json_decode($raw, true);

if (isset($update['message'])) handleMessage($update['message']);
if (isset($update['callback_query'])) handleCallback($update['callback_query']);

http_response_code(200);
echo "OK";
exit;

/* ----------------------------------------------------
    TELEGRAM HELPERS
---------------------------------------------------- */
function tgRequest($method, $params = [])
{
    $url = API_URL . $method;

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

    $res = curl_exec($ch);
    curl_close($ch);
    return $res;
}

function sendMessage($chat, $text, $keyboard = null)
{
    $data = [
        "chat_id" => $chat,
        "text"    => $text,
        "parse_mode" => "HTML"
    ];
    if ($keyboard) $data["reply_markup"] = json_encode($keyboard);

    tgRequest("sendMessage", $data);
}

function editMessage($chat, $msg_id, $text, $keyboard = null)
{
    $data = [
        "chat_id" => $chat,
        "message_id" => $msg_id,
        "text" => $text,
        "parse_mode" => "HTML"
    ];
    if ($keyboard) $data["reply_markup"] = json_encode($keyboard);

    tgRequest("editMessageText", $data);
}

function sendPhoto($chat, $photo, $caption, $keyboard = null)
{
    $data = [
        "chat_id" => $chat,
        "photo"   => $photo,
        "caption" => $caption,
        "parse_mode" => "HTML"
    ];
    if ($keyboard) $data["reply_markup"] = json_encode($keyboard);

    tgRequest("sendPhoto", $data);
}

function answerCallback($id)
{
    tgRequest("answerCallbackQuery", ["callback_query_id" => $id]);
}

function getFileURL($fid)
{
    $json = tgRequest("getFile", ["file_id" => $fid]);
    $j = json_decode($json, true);

    return isset($j["result"]["file_path"])
        ? "https://api.telegram.org/file/bot" . BOT_TOKEN . "/" . $j["result"]["file_path"]
        : "";
}

/* ----------------------------------------------------
     MAIN MESSAGE HANDLER
---------------------------------------------------- */
function handleMessage($msg)
{
    global $pdo;

    $chat = $msg["chat"]["id"];
    $text = $msg["text"] ?? "";
    $name = $msg["from"]["first_name"] ?? "";
    $username = $msg["from"]["username"] ?? "";

    // Save/update user
    $pdo->prepare("
        INSERT INTO users (telegram_id, first_name, username)
        VALUES (?, ?, ?)
        ON DUPLICATE KEY UPDATE first_name=VALUES(first_name), username=VALUES(username)
    ")->execute([$chat, $name, $username]);


    if ($text === "/start") {
        sendMessage($chat, "👋 Welcome $name\nSelect an option:", [
            "keyboard" => [
                [["text" => "🏫 Buy Course"]],
                [["text" => "🧾 Order History"]],
            ],
            "resize_keyboard" => true
        ]);
        return;
    }

    if ($text === "🏫 Buy Course") {
        showInstitutes($chat);
        return;
    }

    if ($text === "🧾 Order History") {
        showOrders($chat);
        return;
    }

    if (isset($msg["photo"])) {
        saveScreenshot($chat, $msg);
        return;
    }

    if (strlen($text) > 5 && is_numeric(str_replace(" ", "", $text))) {
        saveUTR($chat, $text);
        return;
    }

    sendMessage($chat, "❓ I didn’t understand that.\nUse /start");
}
/* ----------------------------------------------------
     LEVEL 1 – INSTITUTES
---------------------------------------------------- */
function showInstitutes($chat)
{
    global $pdo;

    $rows = $pdo->query("SELECT id, name FROM institutes ORDER BY name ASC")->fetchAll();

    if (!$rows) {
        sendMessage($chat, "⚠ No institutes added yet.");
        return;
    }

    $btn = [];
    foreach ($rows as $i) {
        $btn[] = [[
            "text" => $i["name"],
            "callback_data" => "inst_" . $i["id"]
        ]];
    }

    // Back to home
    $btn[] = [[
        "text" => "⬅ Back",
        "callback_data" => "back_home"
    ]];

    sendMessage($chat, "🏫 <b>Select Institute</b>:", ["inline_keyboard" => $btn]);
}

function editShowInstitutes($chat, $msg_id)
{
    global $pdo;

    $rows = $pdo->query("SELECT id, name FROM institutes ORDER BY name ASC")->fetchAll();

    $btn = [];
    foreach ($rows as $i) {
        $btn[] = [[
            "text" => $i["name"],
            "callback_data" => "inst_" . $i["id"]
        ]];
    }

    $btn[] = [[
        "text" => "⬅ Back",
        "callback_data" => "back_home"
    ]];

    editMessage($chat, $msg_id, "🏫 <b>Select Institute</b>:", ["inline_keyboard" => $btn]);
}

/* ----------------------------------------------------
     LEVEL 2 – BRANCHES (EDIT INLINE + BACK)
---------------------------------------------------- */
function editShowBranches($chat, $msg_id, $inst_id)
{
    global $pdo;

    $stmt = $pdo->prepare("SELECT id, name FROM branches WHERE institute_id=? ORDER BY name ASC");
    $stmt->execute([$inst_id]);
    $rows = $stmt->fetchAll();

    // 🔥 If NO BRANCHES → Popup + Auto Back
    if (!$rows) {

        // popup
        tgRequest("answerCallbackQuery", [
    "callback_query_id" => $GLOBALS['last_callback_id'],
    "text" => "⚠ No branches available",
    "show_alert" => false
]);


        // auto go back to institute list
        editShowInstitutes($chat, $msg_id);

        return;
    }

    $btn = [];
    foreach ($rows as $b) {
        $btn[] = [[
            "text" => $b["name"],
            "callback_data" => "branch_" . $b["id"] . "_" . $inst_id
        ]];
    }

    $btn[] = [[
        "text" => "⬅ Back",
        "callback_data" => "back_inst_list"
    ]];

    editMessage($chat, $msg_id, "📚 <b>Select Branch</b>:", ["inline_keyboard" => $btn]);
}


/* ----------------------------------------------------
     LEVEL 3 – COURSES WITH PAGINATION
---------------------------------------------------- */
function editShowCourses($chat, $msg_id, $branch_id, $page = 1, $inst_id = null)
{
    global $pdo;

    $limit = 10;
    $offset = ($page - 1) * $limit;

    $countStmt = $pdo->prepare("SELECT COUNT(*) FROM courses WHERE branch_id=?");
    $countStmt->execute([$branch_id]);
    $total = $countStmt->fetchColumn();

    $stmt = $pdo->prepare("SELECT id, name, price FROM courses WHERE branch_id=? LIMIT $limit OFFSET $offset");
    $stmt->execute([$branch_id]);
    $rows = $stmt->fetchAll();

    // 🔥 NO COURSES → popup + auto-back
    if (!$rows) {
        tgRequest("answerCallbackQuery", [
    "callback_query_id" => $GLOBALS['last_callback_id'],
    "text" => "⚠ No courses available",
    "show_alert" => false
]);


        // back to branches
        editShowBranches($chat, $msg_id, $inst_id);

        return;
    }

    // ---- normal logic below ----

    $btn = [];
    foreach ($rows as $c) {
        $btn[] = [[
            "text" => $c["name"] . " ₹" . $c["price"],
            "callback_data" => "course_" . $c["id"]
        ]];
    }

    $pagination = [];
    if ($page > 1) {
        $pagination[] = [
            "text" => "⬅ Previous",
            "callback_data" => "prev_" . $branch_id . "_" . ($page - 1) . "_" . $inst_id
        ];
    }
    if ($offset + $limit < $total) {
        $pagination[] = [
            "text" => "Next ➡",
            "callback_data" => "next_" . $branch_id . "_" . ($page + 1) . "_" . $inst_id
        ];
    }
    if (!empty($pagination)) $btn[] = $pagination;

    $btn[] = [[
        "text" => "⬅ Back",
        "callback_data" => "back_branch_" . $inst_id
    ]];

    editMessage($chat, $msg_id, "📘 <b>Select Course</b> (Page $page):", [
        "inline_keyboard" => $btn
    ]);
}


/* ----------------------------------------------------
     LEVEL 4 – COURSE DETAILS (EDIT INLINE + BACK)
---------------------------------------------------- */
function editShowCourseDetails($chat, $msg_id, $cid)
{
    global $pdo;

    $stmt = $pdo->prepare("SELECT * FROM courses WHERE id=?");
    $stmt->execute([$cid]);
    $c = $stmt->fetch();

    if (!$c) {
        editMessage($chat, $msg_id, "Course not found.");
        return;
    }

    $msg =
        "<b>{$c['name']}</b>\n\n" .
        "{$c['description']}\n\n" .
        "💰 <b>Price:</b> ₹{$c['price']}";

    $buttons = [
        [["text" => "💳 Buy Now", "callback_data" => "buy_" . $cid]],
        [["text" => "⬅ Back", "callback_data" => "back_course_" . $c["branch_id"] ]]
    ];

    editMessage($chat, $msg_id, $msg, ["inline_keyboard" => $buttons]);
}
/* ----------------------------------------------------
     PAYMENT — CREATE ORDER
---------------------------------------------------- */
function createOrder($chat, $cid)
{
    global $pdo;

    // get user ID
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id=?");
    $stmt->execute([$chat]);
    $u = $stmt->fetch();

    // create pending order
    $pdo->prepare("
        INSERT INTO orders (user_id, course_id, status)
        VALUES (?, ?, 'pending')
    ")->execute([$u["id"], $cid]);

    // Fetch course
    $c = $pdo->prepare("SELECT * FROM courses WHERE id=?");
    $c->execute([$cid]);
    $course = $c->fetch();

    // Load global QR
    $qr = $pdo->query("SELECT value FROM settings WHERE name='global_qr' LIMIT 1")->fetchColumn();

    sendPhoto(
        $chat,
        $qr,
        "💳 Pay ₹{$course['price']} and upload screenshot.\n\nThen send UTR.\n\n⬅ You can go back anytime."
    );
}

/* ----------------------------------------------------
     SAVE SCREENSHOT
---------------------------------------------------- */
function saveScreenshot($chat, $msg)
{
    global $pdo;

    // Get latest order
    $o = $pdo->prepare("
        SELECT id FROM orders
        WHERE user_id=(SELECT id FROM users WHERE telegram_id=?)
        ORDER BY id DESC LIMIT 1
    ");
    $o->execute([$chat]);
    $order = $o->fetch();

    if (!$order) {
        sendMessage($chat, "❌ No active order.");
        return;
    }

    $file_id = end($msg["photo"])["file_id"];
    $url = getFileURL($file_id);

    $pdo->prepare("
        UPDATE orders SET payment_screenshot=?, status='review'
        WHERE id=?
    ")->execute([$url, $order["id"]]);

    sendMessage($chat, "📸 Screenshot received.\nNow send UTR.\n\n⬅ You can go back anytime.");
}

/* ----------------------------------------------------
     SAVE UTR
---------------------------------------------------- */
function saveUTR($chat, $utr)
{
    global $pdo;

    $o = $pdo->prepare("
        SELECT id FROM orders
        WHERE user_id=(SELECT id FROM users WHERE telegram_id=?)
        ORDER BY id DESC LIMIT 1
    ");
    $o->execute([$chat]);
    $order = $o->fetch();

    if (!$order) {
        sendMessage($chat, "❌ No order found.");
        return;
    }

    $pdo->prepare("
        UPDATE orders SET utr=?, status='review'
        WHERE id=?
    ")->execute([$utr, $order["id"]]);

    sendMessage($chat, "⏳ Your payment is under review.\n\n⬅ You can go back anytime.");
}

/* ----------------------------------------------------
     ORDER HISTORY
---------------------------------------------------- */
function showOrders($chat)
{
    global $pdo;

    $stmt = $pdo->prepare("
        SELECT o.id, o.status, c.name
        FROM orders o
        JOIN users u ON o.user_id=u.id
        JOIN courses c ON o.course_id=c.id
        WHERE u.telegram_id=?
        ORDER BY o.id DESC
    ");
    $stmt->execute([$chat]);
    $rows = $stmt->fetchAll();

    if (!$rows) {
        sendMessage($chat, "❌ No orders found.");
        return;
    }

    $msg = "🧾 <b>Your Orders</b>\n\n";
    foreach ($rows as $o) {
        $msg .= "📌 <b>{$o['name']}</b> — <b>{$o['status']}</b>\n";
    }

    sendMessage($chat, $msg);
}

/* ----------------------------------------------------
     CALLBACK ROUTER (FINAL)
---------------------------------------------------- */
function handleCallback($cb)
{
    $GLOBALS['last_callback_id'] = $cb["id"];
    $chat  = $cb["message"]["chat"]["id"];
    $msgId = $cb["message"]["message_id"];
    $data  = $cb["data"];

    /* ===== INSTITUTE HANDLERS ===== */
    if ($data === "back_home") {
        editMessage($chat, $msgId, "👋 Welcome\nSelect an option:", [
            "keyboard" => [
                [["text" => "🏫 Buy Course"]],
                [["text" => "🧾 Order History"]],
            ],
            "resize_keyboard" => true
        ]);
    }
    elseif (strpos($data, "inst_") === 0) {
        editShowBranches($chat, $msgId, substr($data, 5));
    }
    elseif ($data === "back_inst_list") {
        editShowInstitutes($chat, $msgId);
    }

    /* ===== BRANCH HANDLERS ===== */
    elseif (strpos($data, "branch_") === 0) {

    // branch_<branch_id>_<inst_id>
    $parts = explode("_", $data);

    $branch_id = $parts[1];
    $inst_id   = $parts[2];

    editShowCourses($chat, $msgId, $branch_id, 1, $inst_id);
}

    elseif (strpos($data, "back_branch_") === 0) {
        editShowBranches($chat, $msgId, substr($data, 12));
    }

    /* ===== COURSE PAGINATION ===== */
    elseif (strpos($data, "next_") === 0) {
    // next_branch_page_inst
    list(, $branch, $page, $inst_id) = explode("_", $data);
    editShowCourses($chat, $msgId, $branch, $page, $inst_id);
}

elseif (strpos($data, "prev_") === 0) {
    list(, $branch, $page, $inst_id) = explode("_", $data);
    editShowCourses($chat, $msgId, $branch, $page, $inst_id);
}


    /* ===== COURSE DETAILS ===== */
    elseif (strpos($data, "course_") === 0) {
        editShowCourseDetails($chat, $msgId, substr($data, 7));
    }
    elseif (strpos($data, "back_course_") === 0) {
    $branch_id = substr($data, 12);

    // get inst_id from branch
    $stmt = $GLOBALS['pdo']->prepare("SELECT institute_id FROM branches WHERE id=?");
    $stmt->execute([$branch_id]);
    $inst_id = $stmt->fetchColumn();

    editShowCourses($chat, $msgId, $branch_id, 1, $inst_id);
}


    /* ===== BUY ===== */
    elseif (strpos($data, "buy_") === 0) {
        createOrder($chat, substr($data, 4));
    }

    answerCallback($cb["id"]);
}
?>
